/**
 * JavaScript para o Berapay - Admin
 */

(function($) {
    'use strict';

    // Objeto principal do admin
    var WCPixGatewayAdmin = {
        
        init: function() {
            this.bindEvents();
            this.validateForm();
        },
        
        bindEvents: function() {
            // Validação da API Key em tempo real
            $('#api_key').on('input', this.validateApiKey);
            
            // Validação da URL da API
            $('#api_url').on('input', this.validateApiUrl);
            
            // Teste de conexão
            $('#test_api').on('click', this.handleTestConnection);
            
            // Toggle do modo sandbox
            $('#sandbox').on('change', this.handleSandboxToggle);
            
            // Toggle do debug
            $('#debug').on('change', this.handleDebugToggle);
        },
        
        /**
         * Validar API Key
         */
        validateApiKey: function() {
            var $input = $(this);
            var value = $input.val().trim();
            var $feedback = $input.siblings('.api-key-feedback');
            
            // Remover feedback anterior
            $feedback.remove();
            
            if (value.length === 0) {
                return;
            }
            
            var isValid = value.length >= 20; // Assumindo que uma API key válida tem pelo menos 20 caracteres
            var feedbackClass = isValid ? 'valid' : 'invalid';
            var feedbackText = isValid ? 'API Key parece válida' : 'API Key muito curta';
            
            $input.after('<span class="api-key-feedback ' + feedbackClass + '">' + feedbackText + '</span>');
        },
        
        /**
         * Validar URL da API
         */
        validateApiUrl: function() {
            var $input = $(this);
            var value = $input.val().trim();
            var $feedback = $input.siblings('.api-url-feedback');
            
            // Remover feedback anterior
            $feedback.remove();
            
            if (value.length === 0) {
                return;
            }
            
            try {
                var url = new URL(value);
                var isValid = url.protocol === 'https:' && url.hostname.includes('getpaybr.com');
                var feedbackClass = isValid ? 'valid' : 'invalid';
                var feedbackText = isValid ? 'URL válida' : 'URL deve ser HTTPS e do domínio correto';
                
                $input.after('<span class="api-url-feedback ' + feedbackClass + '">' + feedbackText + '</span>');
            } catch (e) {
                $input.after('<span class="api-url-feedback invalid">URL inválida</span>');
            }
        },
        
        /**
         * Gerenciar teste de conexão
         */
        handleTestConnection: function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var $form = $button.closest('form');
            var originalText = $button.val();
            
            // Validar campos obrigatórios
            var apiKey = $('#api_key').val().trim();
            var apiUrl = $('#api_url').val().trim();
            
            if (!apiKey || !apiUrl) {
                alert('Por favor, preencha a API Key e URL antes de testar a conexão.');
                return;
            }
            
            // Estado de loading
            $button.val('Testando...').prop('disabled', true);
            
            // Simular delay mínimo para feedback visual
            setTimeout(function() {
                $form.submit();
            }, 500);
        },
        
        /**
         * Toggle do modo sandbox
         */
        handleSandboxToggle: function() {
            var $checkbox = $(this);
            var isChecked = $checkbox.is(':checked');
            var $notice = $('.sandbox-notice');
            
            if (isChecked) {
                if ($notice.length === 0) {
                    $checkbox.closest('td').append(
                        '<div class="sandbox-notice" style="margin-top: 10px; padding: 10px; background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 4px; color: #856404;">' +
                        '<strong>Modo Teste Ativado:</strong> Nenhuma transação real será processada.' +
                        '</div>'
                    );
                }
            } else {
                $notice.remove();
            }
        },
        
        /**
         * Toggle do debug
         */
        handleDebugToggle: function() {
            var $checkbox = $(this);
            var isChecked = $checkbox.is(':checked');
            var $notice = $('.debug-notice');
            
            if (isChecked) {
                if ($notice.length === 0) {
                    $checkbox.closest('td').append(
                        '<div class="debug-notice" style="margin-top: 10px; padding: 10px; background: #d1ecf1; border: 1px solid #bee5eb; border-radius: 4px; color: #0c5460;">' +
                        '<strong>Debug Ativado:</strong> Logs detalhados serão gerados. Verifique em WooCommerce > Status > Logs.' +
                        '</div>'
                    );
                }
            } else {
                $notice.remove();
            }
        },
        
        /**
         * Validação geral do formulário
         */
        validateForm: function() {
            var $form = $('form');
            
            $form.on('submit', function(e) {
                var isValid = true;
                var errors = [];
                
                // Validar API Key
                var apiKey = $('#api_key').val().trim();
                if (apiKey.length > 0 && apiKey.length < 20) {
                    errors.push('API Key parece ser muito curta');
                    isValid = false;
                }
                
                // Validar URL
                var apiUrl = $('#api_url').val().trim();
                if (apiUrl.length > 0) {
                    try {
                        var url = new URL(apiUrl);
                        if (url.protocol !== 'https:') {
                            errors.push('URL da API deve usar HTTPS');
                            isValid = false;
                        }
                    } catch (e) {
                        errors.push('URL da API é inválida');
                        isValid = false;
                    }
                }
                
                if (!isValid) {
                    e.preventDefault();
                    alert('Erros encontrados:\n' + errors.join('\n'));
                    return false;
                }
            });
        },
        
        /**
         * Exibir notificação
         */
        showNotice: function(message, type) {
            type = type || 'info';
            
            var $notice = $('<div class="notice notice-' + type + ' is-dismissible"><p>' + message + '</p></div>');
            
            $('.wrap h1').after($notice);
            
            // Auto-remover após 5 segundos
            setTimeout(function() {
                $notice.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);
        }
    };
    
    // Inicializar quando DOM estiver pronto
    $(document).ready(function() {
        if ($('body').hasClass('woocommerce_page_wc-pix-gateway-settings') || 
            ($('#api_key').length && $('#api_url').length)) {
            WCPixGatewayAdmin.init();
        }
    });
    
    // Expor para uso global se necessário
    window.WCPixGatewayAdmin = WCPixGatewayAdmin;
    
})(jQuery);
